package easik.states;

import java.util.ArrayList;

import javax.swing.JOptionPane;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.SumConstraint;
import easik.sketch.path.SketchPath;


/** 
 * This state makes a sum constraint. When pushed, it will then push another
 * state to get a path. Once the first path is received, it will push another request for
 * a path. After adding the first path the options to either add another path or finish
 * the constraint are available.
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @since 2006-05-23 Kevin Green
 * @version 2006-08-15 Kevin Green
 */
public class AddSumConstraintState extends EasikState implements PathAcceptingState {
	/**
	 * The paths involved in this constraint
	 */
	private ArrayList<SketchPath> _paths;
	/**
	 * Stores whether the user has finished adding paths to this constraint
	 */
	private boolean _finished;


	/**
	 * Constructor for creating a new state to collect paths and make a diagram.
	 * 
	 * @param inSketch The sketch being worked with.
	 */
	public AddSumConstraintState(Sketch inSketch) {
		_ourSketch = inSketch;
		_finished = false;
		_paths = new ArrayList<SketchPath>();
	}
	
	/**
	 * Called when a new path has been completed by a state above this on the
	 * stack. If it is null, then cancel, otherwise check to see if the user
	 * selected the 'next' or 'finish' option.
	 * 
	 * @param inPath The last path in, null if it was a cancellation
	 */
	public void passPath(SketchPath inPath) {
		_finished = Easik.getInstance().getStateManager().getFinished();
		//Check if cancelled
		if (inPath == null) {
			Easik.getInstance().getStateManager().popState();
			return;
		}
		
		//Check which path is being passed
		if(!_finished){
			_paths.add(inPath);
			Easik.getInstance().getStateManager().pushState(new GetPathState(true, true));
		}
		
		else{
			_paths.add(inPath);
			if(addDiagram()){
				JOptionPane.showMessageDialog(
					_ourSketch.getParent(),
					"Created a Sum Constraint",
					"JEase",
					JOptionPane.INFORMATION_MESSAGE);
			}
			else{
				JOptionPane.showMessageDialog(
						Easik.getInstance().getFrame(),
						"Sum constraint could not be created.\n"
							+ "Please ensure that:\n"
							+ "  1) All edges involved target the same entity\n"
							+ "  2) The first edge in each path is injective\n"
							+ "  3) At least two edges are selected\n"
							+ "  4) All paths are unique",
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
			Easik.getInstance().getStateManager().popState();
		}
		
	}

	/**
	 * Add the diagram to the sketch
	 * @return true if the diagram was successfully added, false otherwise
	 */
	private boolean addDiagram() {
		if(SumConstraint.isSumConstraint(_paths)){
			Constraint newDiagram = new SumConstraint(_paths);
			Easik.getInstance().getFrame().getSketch().addNewConstraint(newDiagram);
			Easik.getInstance().getFrame().getSketch().setDirty(true);
			return true;
		}
		return false;
	}


	/**
	 * When this state is pushed on, it sends a message in a popup and then pushes
	 * a path collection state.
	 */
	public void pushedOn() {
		Easik.getInstance().getFrame().getSketch().getSelectionListener().emptySelection();
		Easik.getInstance().getStateManager().resetFinished();
		JOptionPane.showMessageDialog(
			_ourSketch.getParent(),
			"Select a path and then press 'Next' to add it to the constraint and 'Finish' when you are done adding paths",
			"JEase",
			JOptionPane.INFORMATION_MESSAGE);

		Easik.getInstance().getFrame().setStateString("Sum Constraint");
		setNextButton(true);
		setCancelButton(true);
		setFinishButton(false);
		Easik.getInstance().getStateManager().pushState(new GetPathState(true, false));
	}

	/**
	 * Nothing to do when it gets popped off, as it always pops itself off when completed. 
	 */
	public void poppedOff() {
	}

	/**
	 * This state is called "New Sum Constraint".
	 * 
	 * @return The string literal described above.
	 */
	public String toString() {
		return "New Sum Constraint";
	}

}

